/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: David Papa <iamyou@umich.edu>

ChangeLog:
2004-09-15: ChangeLog started
 */

#include <iostream>
#include <string>
#include <sstream>


#include <qaction.h>
#include <qapplication.h>
#include <qlabel.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qstatusbar.h>
#include <qworkspace.h>
#include <qvbox.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qobjectlist.h>
#include <qpixmap.h>

#include "mainwindow.h"
#include "oagConnect.h"
#include "SchemInstView/SchemInstViewWnd.h"
//#include "PlacerPlugin.h"
#include "DebugLog.h"
#include "config.h"
#include "oagThread.h"
#include "QoagCommon/QoagEvents.h"

using std::cerr;
using std::endl;
using std::string;
using std::stringstream;

using oa::oaInst;
using oa::oaNativeNS;
using oa::oaSimpleName;

namespace oagBazaar
{

    MainWindow::MainWindow(QWidget *parent, const char *name, Controller& controllerIn)
        : QMainWindow(parent, name), controller(controllerIn)
        {
            workspace = new QWorkspace(this);
            workspace->setScrollBarsEnabled(true);
            setCentralWidget(workspace);
            connect(workspace, SIGNAL(windowActivated(QWidget *)),
                    this, SLOT(updateMenus()));
            connect(workspace, SIGNAL(windowActivated(QWidget *)),
                    this, SLOT(updateModIndicator()));

            openDialog = new openDesignFormT(this, "openDesignForm", TRUE);
            connect(openDialog, SIGNAL(sigDesignLoad()),
                    this, SLOT(loadLayout()));

            openSchemDialog = new openSchematicFormT(this, "openSchemDialog", TRUE);
            connect(openSchemDialog, SIGNAL(sigSchematicLoad()),
                    this, SLOT(loadSchematic()));

            openOADBDialog = new openOADBForm(this, "openOADBDialog", TRUE);
            connect(openOADBDialog, SIGNAL(openOADBPressed(const std::string&)),
                    this, SLOT(loadLibraryDefinitionFile(const std::string&)));

            loadPlugInDialog = new loadPlugInFormT(this, "loadPlugInDialog", TRUE);
            connect(loadPlugInDialog, SIGNAL(loadPlugInPressed(const std::string&)),
                    this, SLOT(loadPlugInFile(const std::string&)));

            saveAsDialog = new saveDesignAsFormT(this, "saveDesignAsForm", TRUE);
            connect(saveAsDialog, SIGNAL(sigDesignSave()),
                    this, SLOT(doSaveAs()));


            createActions();
            createMenus();
            createToolBars();
            createStatusBar();
            createCommandWindow();

            //QoaCommandDockWindow* bottomDockCommandWindow = new QoaCommandDockWindow(&mainWin, controller);    
            //mainWin.moveDockWindow(bottomDockCommandWindow, Qt::DockBottom);
            //mainWin.show();

            setCaption(tr("OAGear Bazaar"));
            setIcon(QPixmap::fromMimeSource("icon.png"));
        }

    void MainWindow::createActions()
    {
        //newAct = new QAction(tr("&New"), tr("Ctrl+N"), this);
        //newAct->setIconSet(QPixmap::fromMimeSource("new.png"));
        //newAct->setStatusTip(tr("Create a new file"));
        //connect(newAct, SIGNAL(activated()), this, SLOT(newFile()));

        openLayoutAct = new QAction(tr("&Open Layout..."), tr("Ctrl+O"), this);
        openLayoutAct->setIconSet(QPixmap::fromMimeSource("open.png"));
        openLayoutAct->setStatusTip(tr("Open a design in layout view"));
        connect(openLayoutAct, SIGNAL(activated()), this, SLOT(open()));

        openSchemAct = new QAction(tr("Open Sc&hematic..."), tr("Ctrl+H"), this);
        openSchemAct->setIconSet(QPixmap::fromMimeSource("open.png"));
        openSchemAct->setStatusTip(tr("Open a design in schematic view"));
        connect(openSchemAct, SIGNAL(activated()), this,  SLOT(openSchem()));

        openOADBAct = new QAction(tr("Open OpenAccess Database..."), 0, this);
        openOADBAct->setIconSet(QPixmap::fromMimeSource("oadb_icon.png"));
        openOADBAct->setStatusTip(tr("Open an OpenAccess Database from lib.defs file"));
        connect(openOADBAct, SIGNAL(activated()), this,  SLOT(openOADB()));

        loadPlugInAct = new QAction(tr("Load Bazaar PlugIn..."), 0, this);
        loadPlugInAct->setIconSet(QPixmap::fromMimeSource("plug_icon.png"));
        loadPlugInAct->setStatusTip(tr("Load Bazaar PlugIn from shared object file (.so)"));
        connect(loadPlugInAct, SIGNAL(activated()), this,  SLOT(loadPlugIn()));

        saveAct = new QAction(tr("&Save"), tr("Ctrl+S"), this);
        saveAct->setIconSet(QPixmap::fromMimeSource("save.png"));
        saveAct->setStatusTip(tr("Save the file to disk"));
        connect(saveAct, SIGNAL(activated()), this, SLOT(save()));

        saveAsAct = new QAction(tr("Save &As..."), 0, this);
        saveAsAct->setIconSet(QPixmap::fromMimeSource("save.png"));
        saveAsAct->setStatusTip(tr("Save the file under a new name"));
        connect(saveAsAct, SIGNAL(activated()), this, SLOT(saveAs()));

        exitAct = new QAction(tr("E&xit"), tr("Ctrl+Q"), this);
        exitAct->setStatusTip(tr("Exit the application"));
        connect(exitAct, SIGNAL(activated()), this, SLOT(close()));

        cutAct = new QAction(tr("Cu&t"), tr("Ctrl+X"), this);
        cutAct->setIconSet(QPixmap::fromMimeSource("cut.png"));
        cutAct->setStatusTip(tr("Cut the current selection to the "
                    "clipboard"));
        connect(cutAct, SIGNAL(activated()), this, SLOT(cut()));

        copyAct = new QAction(tr("&Copy"), tr("Ctrl+C"), this);
        copyAct->setIconSet(QPixmap::fromMimeSource("copy.png"));
        copyAct->setStatusTip(tr("Copy the current selection to the "
                    "clipboard"));
        connect(copyAct, SIGNAL(activated()), this, SLOT(copy()));

        pasteAct = new QAction(tr("&Paste"), tr("Ctrl+V"), this);
        pasteAct->setIconSet(QPixmap::fromMimeSource("paste.png"));
        pasteAct->setStatusTip(tr("Paste the clipboard's contents at "
                    "the cursor position"));
        connect(pasteAct, SIGNAL(activated()), this, SLOT(paste()));

        deleteAct = new QAction(tr("&Delete"), tr("Del"), this);
        deleteAct->setIconSet(QPixmap::fromMimeSource("delete.png"));
        deleteAct->setStatusTip(tr("Delete the current selection"));
        connect(deleteAct, SIGNAL(activated()), this, SLOT(del()));

        closeAct = new QAction(tr("Cl&ose"), tr("Ctrl+F4"), this);
        closeAct->setStatusTip(tr("Close the active window"));
        connect(closeAct, SIGNAL(activated()),
                workspace, SLOT(closeActiveWindow()));

        closeAllAct = new QAction(tr("Close &All"), 0, this);
        closeAllAct->setStatusTip(tr("Close all the windows"));
        connect(closeAllAct, SIGNAL(activated()),
                workspace, SLOT(closeAllWindows()));

        tileAct = new QAction(tr("&Tile"), 0, this);
        tileAct->setStatusTip(tr("Tile the windows"));
        connect(tileAct, SIGNAL(activated()), workspace, SLOT(tile()));

        cascadeAct = new QAction(tr("&Cascade"), 0, this);
        cascadeAct->setStatusTip(tr("Cascade the windows"));
        connect(cascadeAct, SIGNAL(activated()),
                workspace, SLOT(cascade()));

        nextAct = new QAction(tr("Ne&xt"), tr("Ctrl+F6"), this);
        nextAct->setStatusTip(tr("Move the focus to the next window"));
        connect(nextAct, SIGNAL(activated()),
                workspace, SLOT(activateNextWindow()));

        previousAct = new QAction(tr("Pre&vious"),
                tr("Ctrl+Shift+F6"), this);
        previousAct->setStatusTip(tr("Move the focus to the previous "
                    "window"));
        connect(previousAct, SIGNAL(activated()),
                workspace, SLOT(activatePreviousWindow()));

        aboutAct = new QAction(tr("&About"), 0, this);
        aboutAct->setStatusTip(tr("Show the application's About box"));
        connect(aboutAct, SIGNAL(activated()), this, SLOT(about()));

        aboutQtAct = new QAction(tr("About &Qt"), 0, this);
        aboutQtAct->setStatusTip(tr("Show the Qt library's About box"));
        connect(aboutQtAct, SIGNAL(activated()), qApp, SLOT(aboutQt()));

        zoomInAct = new QAction(tr("Zoom In"),0,this);
        zoomInAct->setStatusTip(tr("Zoom In"));
        zoomInAct->setIconSet(QPixmap::fromMimeSource("zoom_in.png"));
        connect(zoomInAct, SIGNAL(activated()), this, SLOT(zoomIn()));

        zoomOutAct = new QAction(tr("Zoom Out"),0,this);
        zoomOutAct->setStatusTip(tr("Zoom Out"));
        zoomOutAct->setIconSet(QPixmap::fromMimeSource("zoom_out.png"));
        connect(zoomOutAct, SIGNAL(activated()), this, SLOT(zoomOut()));    

        //runSchemAct = new QAction(tr("Schematic"),0,this);
        //runSchemAct->setStatusTip(tr("Run Schematic Viewer"));
        //runSchemAct->setIconSet(QPixmap::fromMimeSource("designer_editgrid.png"));
        //connect(runSchemAct, SIGNAL(activated()), this, SLOT(addSchematic()));    
    }

    void MainWindow::createMenus()
    {
        fileMenu = new QPopupMenu(this);
        //newAct->addTo(fileMenu);
        openLayoutAct->addTo(fileMenu);
        openSchemAct->addTo(fileMenu);     
        openOADBAct->addTo(fileMenu);
        loadPlugInAct->addTo(fileMenu);
        saveAct->addTo(fileMenu);
        saveAsAct->addTo(fileMenu);
        fileMenu->insertSeparator();
        exitAct->addTo(fileMenu);

        //    editMenu = new QPopupMenu(this);
        //    cutAct->addTo(editMenu);
        //    copyAct->addTo(editMenu);
        //    pasteAct->addTo(editMenu);
        //    deleteAct->addTo(editMenu);
        viewMenu = new QPopupMenu(this);
        zoomInAct->addTo(viewMenu);
        zoomOutAct->addTo(viewMenu);


        windowsMenu = new QPopupMenu(this);
        createWindowsMenu();

        helpMenu = new QPopupMenu(this);
        aboutAct->addTo(helpMenu);
        aboutQtAct->addTo(helpMenu);

        menuBar()->insertItem(tr("&File"), fileMenu);
        //   menuBar()->insertItem(tr("&Edit"), editMenu);
        menuBar()->insertItem(tr("&View"), viewMenu);
        menuBar()->insertItem(tr("&Windows"), windowsMenu);
        menuBar()->insertSeparator();
        menuBar()->insertItem(tr("&Help"), helpMenu);
    }

    QPopupMenu* MainWindow::addMenu(const string& name)
    {
        QPopupMenu* newMenu = new QPopupMenu(this);
        menuBar()->insertItem(tr(name.c_str()), newMenu);
        return newMenu;
    }

    QPopupMenu* MainWindow::addMenu(const string& iconName, const string& name)
    {
        //get rid of unused compiler warning
        static_cast<void>(name);
        QPopupMenu* newMenu = new QPopupMenu(this);
        menuBar()->insertItem(QPixmap::fromMimeSource(iconName.c_str()) /*, tr(name.c_str()) */ , newMenu);
        return newMenu;
    }

    QToolBar* MainWindow::addToolBar(const string& name)
    {
        QToolBar* newToolBar = new QToolBar(tr(name.c_str()), this);
        return newToolBar;
    }

    void MainWindow::takeWidget(QWidget* widget)
    {
        int width = widget->width(), height = widget->height();
        int tlx = workspace->width() - width, tly = workspace->height() - height;
        tlx = std::max(tlx, 1);
        tly = std::max(tly, 1);
        widget->reparent(workspace, QPoint(rand() % tlx, rand() % tly));
        widget->show();
    }

    void MainWindow::takeDialog(QDialog* dialog)
    {
        int width = dialog->width(), height = dialog->height();
        int tlx = workspace->width() - width, tly = workspace->height() - height;
        tlx = std::max(tlx, 1);
        tly = std::max(tly, 1);
        //dialog->reparent(workspace, QPoint(rand() % tlx, rand() % tly));
        dialog->exec();
    }
    void MainWindow::takeThread(QoagThread* thread)
    {
        //dialog->reparent(workspace, QPoint(rand() % tlx, rand() % tly));
        ProcessManager& processManager = controller.getProcessManager();
        thread->setProcessManager(&processManager);
        processManager.insert(thread);
    }

    void MainWindow::registerCommand(Tcl_CmdProc* cmd, const string& cmdName, ClientData clientData, Tcl_CmdDeleteProc *deleteProc)
    {
        controller.registerCommand(cmd, cmdName, clientData, deleteProc);
    }

    void MainWindow::registerAction(const string& command, QAction* action)
    {
        _actionToCommand[action]=command;
    }

    void MainWindow::createToolBars()
    {
        fileToolBar = new QToolBar(tr("File"), this);
        //newAct->addTo(fileToolBar);
        openLayoutAct->addTo(fileToolBar);
        openSchemAct->addTo(fileToolBar);
        openOADBAct->addTo(fileToolBar);
        loadPlugInAct->addTo(fileToolBar);
        saveAct->addTo(fileToolBar);
        saveAsAct->addTo(fileToolBar);

        editToolBar = new QToolBar(tr("Edit"), this);
        // cutAct->addTo(editToolBar);
        // copyAct->addTo(editToolBar);
        // pasteAct->addTo(editToolBar);
        zoomInAct->addTo(editToolBar);
        zoomOutAct->addTo(editToolBar);


        //runSchemAct->addTo(editToolBar);
    }

    void MainWindow::zoomIn()
    {
        if(activeLayoutViewWnd())
            activeLayoutViewWnd()->zoom_in();
    }

    void MainWindow::zoomOut()
    {
        if(activeLayoutViewWnd())
            activeLayoutViewWnd()->zoom_out();
    }


    void MainWindow::createStatusBar()
    {
        readyLabel = new QLabel(tr(" Ready"), this);

        modLabel = new QLabel(tr(" MOD "), this);
        modLabel->setAlignment(AlignHCenter);
        modLabel->setMinimumSize(modLabel->sizeHint());
        modLabel->clear();

        statusBar()->addWidget(readyLabel, 1);
        statusBar()->addWidget(modLabel);
    }

    void MainWindow::createCommandWindow()
    {
        QWidget * bottomDockCommandWindowParent = this;
        bottomDockCommandWindow = 
            new QoaCommandDockWindow(bottomDockCommandWindowParent, controller);    
        moveDockWindow(bottomDockCommandWindow, Qt::DockBottom);
    }


    void MainWindow::newFile()
    {
        //    LayoutViewWnd *design = createLayoutViewWnd();
        //    design->show();
    }


    LayoutViewWnd* MainWindow::createLayoutViewWnd(oagConnection* dbConnection)
    {
        LayoutViewWnd* design = new LayoutViewWnd(workspace, dbConnection);
        //connect(design, SIGNAL(copyAvailable(bool)),
        ///       this, SLOT(copyAvailable(bool)));
        //connect(editor, SIGNAL(modificationChanged(bool)),
        ///       this, SLOT(updateModIndicator()));
        return design;
    }

    void MainWindow::open()
    {
        openDialog->readLibs();
        openDialog->exec();
    }

    void MainWindow::openSchem()
    {
        openSchemDialog->readLibs();
        openSchemDialog->exec();
    }

    void MainWindow::openOADB()
    {
       openOADBDialog->exec();
    }

    void MainWindow::loadPlugIn()
    {
       loadPlugInDialog->exec();
    }

    void MainWindow::openFile(const string &fileName)
    { 
        //remove compiler unused warning
        static_cast<void>(fileName); 

        openDialog->readLibs();
        openDialog->exec();
    }

    void MainWindow::loadLayout(void)
    {
        QString   q_lib  = openDialog->libComboBox->currentText(),
                  q_cell = openDialog->cellComboBox->currentText(),
                  q_view = openDialog->viewComboBox->currentText();
        if(q_lib.isEmpty() || q_cell.isEmpty() || q_view.isEmpty())
            return; //invalid input

        string lib(q_lib.ascii()),
               cell(q_cell.ascii()),
               view(q_view.ascii());

        string load_layout_command = "load_layout " + lib + " " + cell + " " + view;

        controller.runCommand(load_layout_command);
    }

    void MainWindow::loadSchematic(void)
    {
        QString   q_lib = openSchemDialog->libComboBox->currentText(),
                  q_cell = openSchemDialog->cellComboBox->currentText(),
                  q_view = openSchemDialog->viewComboBox->currentText();

        if(q_lib.isEmpty() || q_cell.isEmpty() || q_view.isEmpty())
            return; //invalid input

        string lib(q_lib.ascii()),
               cell(q_cell.ascii()),
               view(q_view.ascii());

        string load_schematic_command = "load_schematic " + lib + " " + cell + " " + view;

        controller.runCommand(load_schematic_command);
    }

    void MainWindow::loadLibraryDefinitionFile(const std::string& fileName)
    {
       string libraryDefinitionFileCommand = "library_definition_file " + fileName;
       controller.runCommand(libraryDefinitionFileCommand);
    }

    void MainWindow::loadPlugInFile(const std::string& fileName)
    {
       string loadPlugInFileCommand = "load_plugin " + fileName;
       controller.runCommand(loadPlugInFileCommand);
    }

    void MainWindow::save()
    {
        LayoutViewWnd* layoutWnd = activeLayoutViewWnd(); 
        if (layoutWnd) {
            //activeEditor()->save();
            string save_cmd = "save "+ layoutWnd->get_lib() + " " + layoutWnd->get_cell() + " " + layoutWnd->get_view();

            controller.runCommand(save_cmd);
            layoutWnd->setUnmodified();
            //     updateModIndicator();
        }

    }

    LayoutViewWnd* MainWindow::activeLayoutViewWnd()
    {
        LayoutViewWnd* layoutWnd = dynamic_cast<LayoutViewWnd*>(workspace->activeWindow());
        return layoutWnd;
    }

    void MainWindow::saveAs()
    {
        saveAsDialog->readLibs();
        saveAsDialog->exec();
    }

    void MainWindow::doSaveAs()
    {
        cerr<<"DPDEBUG: checking saveAs"<<endl;
        /*
           if (activeLayoutViewWnd()) {
           activeLayoutViewWnd()->saveAs();
           updateModIndicator();
           }
         */

        LayoutViewWnd* layoutWnd = activeLayoutViewWnd(); 
        if (!layoutWnd)
        {
            cerr<<"DPDEBUG: giving up saveAs because there is no active layout window"<<endl;
            return;
        }

        //activeEditor()->save();
        string old_lib = layoutWnd->get_lib(),
               old_cell = layoutWnd->get_cell(),
               old_view = layoutWnd->get_view();


        QString   q_lib = saveAsDialog->libComboBox->currentText(),
                  q_cell = saveAsDialog->cellComboBox->currentText(),
                  q_view = saveAsDialog->viewComboBox->currentText();

        if(q_lib.isEmpty() || q_cell.isEmpty() || q_view.isEmpty())
            return; //invalid input

        string new_lib(q_lib.ascii()),
               new_cell(q_cell.ascii()),
               new_view(q_view.ascii());

        string save_as_command = "save_as " + old_lib + " " + old_cell + " " + old_view + " " + new_lib + " " + new_cell + " " + new_view;

        controller.runCommand(save_as_command);

        layoutWnd->setUnmodified();
    } 

    void MainWindow::cut()
    {
        /*
           if (activeLayoutViewWnd())
           activeLayoutViewWnd()->cut();
         */
    }

    void MainWindow::copy()
    {
        /*
           if (activeLayoutViewWnd())
           activeLayoutViewWnd()->copy();
         */
    }

    void MainWindow::paste()
    {
        /*
           if (activeLayoutViewWnd())
           activeLayoutViewWnd()->paste();
         */
    }

    void MainWindow::del()
    {
        /*
           if (activeLayoutViewWnd())
           activeLayoutViewWnd()->del();
         */
    }

    void MainWindow::about()
    {
        stringstream ss;
        ss << "<h2>OAGear Bazaar " << VERSION << " </h2>";
        ss << "<p>Copyright &copy; 2004-2005 Cadence Design Systems Inc.";
        ss << "<p>OAGear Bazaar is a GUI front end for <b>OpenAccess</b> and its ";
        ss << "many tools.";
        QMessageBox::about(this, tr("About OAGear Bazaar"),
                tr(ss.str().c_str()));
    }

    void MainWindow::updateMenus()
    {
        bool hasEditor = (activeLayoutViewWnd() != 0);
        saveAct->setEnabled(hasEditor);
        saveAsAct->setEnabled(hasEditor);
        pasteAct->setEnabled(hasEditor);
        deleteAct->setEnabled(hasEditor);
        //copyAvailable(activeLayoutViewWnd() && activeLayoutViewWnd()->hasSelectedText());
        closeAct->setEnabled(hasEditor);
        closeAllAct->setEnabled(hasEditor);
        tileAct->setEnabled(hasEditor);
        cascadeAct->setEnabled(hasEditor);
        nextAct->setEnabled(hasEditor);
        previousAct->setEnabled(hasEditor);

        windowsMenu->clear();
        createWindowsMenu();
    }

    void MainWindow::createWindowsMenu()
    {
        closeAct->addTo(windowsMenu);
        closeAllAct->addTo(windowsMenu);
        windowsMenu->insertSeparator();
        tileAct->addTo(windowsMenu);
        cascadeAct->addTo(windowsMenu);
        windowsMenu->insertSeparator();
        nextAct->addTo(windowsMenu);
        previousAct->addTo(windowsMenu);

        if (activeLayoutViewWnd()) {
            windowsMenu->insertSeparator();
            windows = workspace->windowList();
            int numVisibleEditors = 0;

            for (int i = 0; i < (int)windows.count(); ++i) {
                QWidget *win = windows.at(i);
                if (!win->isHidden()) {
                    QString text = tr("%1 %2")
                        .arg(numVisibleEditors + 1)
                        .arg(win->caption());
                    if (numVisibleEditors < 9)
                        text.prepend("&");
                    int id = windowsMenu->insertItem(
                            text, this, SLOT(activateWindow(int)));
                    bool isActive = (activeLayoutViewWnd() == win);
                    windowsMenu->setItemChecked(id, isActive);
                    windowsMenu->setItemParameter(id, i);
                    ++numVisibleEditors;
                }
            }
        }
    }

    void MainWindow::activateWindow(int param)
    {
        QWidget *win = windows.at(param);
        win->show();
        win->setFocus();
    }

    void MainWindow::copyAvailable(bool available)
    {
        cutAct->setEnabled(available);
        copyAct->setEnabled(available);
    }

    void MainWindow::updateModIndicator()
    {
        /*
           if (activeLayoutViewWnd() && activeLayoutViewWnd()->isModified())
           modLabel->setText(tr("MOD"));
           else
           modLabel->clear();*/
    }

    void MainWindow::closeEvent(QCloseEvent *event)
    {
        workspace->closeAllWindows();
        if (activeLayoutViewWnd())
            event->ignore();
        else
            event->accept();
    }

    Qoag::QoaSCVScrollingFrame* MainWindow::createSchematic(const string& lib, const string& cell, const string& view)
    {
        Qoag::QoaSCVScrollingFrame* scvf = new Qoag::QoaSCVScrollingFrame(workspace, lib, cell, view);
        DebugLog::ostream()<<"DPMAINWINDOWDEBUG: I am trying to create a SchemCellView"<<endl;
        return scvf;
    }

    void MainWindow::loadSchematic(const string& lib, const string& cell, const string& view)
    {
        Qoag::QoaSCVScrollingFrame* scvf = createSchematic(lib, cell, view);    
        DebugLog::ostream()<<"DPMAINWINDOWDEBUG: created a (SchemInstView) Qoag::QoaSCVScrollingFrame* "<<scvf<<" from a public function."<<endl; 
        scvf->show();
    }

    void MainWindow::loadLayout(oagConnection* dbConnection)
    {
        LayoutViewWnd* myDesign = createLayoutViewWnd(dbConnection);
        DebugLog::ostream()<<"MAINWINDOWDEBUG: created a LayoutViewWnd* "<<myDesign<<" from a public function."<<endl; 
        myDesign->show();
    }

    commandWindowT* MainWindow::getCommandWindow(void)
    {
        return bottomDockCommandWindow->getCommandWindow();
    }

    void MainWindow::closeActiveWindow()
    {
        LayoutViewWnd* loWnd = activeLayoutViewWnd();
        if ( loWnd )
            loWnd->close();
    }

    LayoutViewWnd* MainWindow::getLayoutWnd(const string& lib, const string& cell, const string view)
    {
        QWidgetList wlist = workspace->windowList();
        LayoutViewWnd* rval=0;
        QWidget* obj=0;
        for( obj = wlist.first(); obj; obj = wlist.next())
        { 
            DebugLog::ostream()<<"MAINWINDOWDEBUG: Searching with obj = "<<obj<<endl;
            // for each found object...
            LayoutViewWnd* curWnd = dynamic_cast<LayoutViewWnd*>(obj);
            if(curWnd)
            {
                DebugLog::ostream()<<"MAINWINDOWDEBUG: Found with LayoutViewWnd with Lib: "<<curWnd->get_lib()<<" Cell: "<<curWnd->get_cell()<<" View: "<<curWnd->get_view()<<endl;
                if( curWnd->get_lib() == lib &&
                        curWnd->get_cell() == cell &&
                        curWnd->get_view() == view)
                {
                    rval = curWnd;
                    return rval;
                }
            }
        }
        return rval;
    }

    void MainWindow::placingLayout(const string& lib, const string& cell, const string view)
    {
        LayoutViewWnd* loWnd = getLayoutWnd(lib, cell, view);
        if(!loWnd)
        { 
            DebugLog::ostream()<<"MAINWINDOWDEBUG: placingLayout slot called, but no such window found"<<endl;
            return;
        }
        loWnd->showLabel("Placing...");
        loWnd->refresh();
    }
    void MainWindow::placedLayout(const string& lib, const string& cell, const string view)
    {
        LayoutViewWnd* loWnd = getLayoutWnd(lib, cell, view);
        if(!loWnd)
        { 
            DebugLog::ostream()<<"MAINWINDOWDEBUG: placedLayout slot called, but no such window found"<<endl;
            return;
        }
        loWnd->hideLabel();
        loWnd->setModified();
        loWnd->refresh();
    }

    void MainWindow::runCommand( QAction* action)
    {
        std::map<QAction*, string>::iterator actionIt = _actionToCommand.find(action);
        if(actionIt != _actionToCommand.end())
            if(!actionIt->second.empty())
                controller.runCommand(actionIt->second);
            else
                cerr<<"Run command called from an unknown action, ignoring."<<endl;

    }

    void MainWindow::runCommand(const std::string& command)
    {
        if(!command.empty())
            controller.runCommand(command);
    }

    void MainWindow::getActiveDesign(string& library, string& cell, string& view) const
    {
        library="";
        cell="";
        view="";
        LayoutViewWnd* layoutWnd = dynamic_cast<LayoutViewWnd*>(workspace->activeWindow());
        if(layoutWnd)
        {
            library = layoutWnd->get_lib();
            cell = layoutWnd->get_cell();
            view = layoutWnd->get_view();
            cout<<"In mainwindow setting design to "<<library<<"/"<<cell<<"/"<<view<<endl;
        }
    }

}



