/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */
#include <iostream>
#include <iomanip>

#include <qtable.h>
#include <qlineedit.h>

#include "capoPlugIn.h"
#include "QoagCommon/QoagCommonTempl.h"
#include "PlacerInterface.h"
#include "runCapoOptions.h"
#include "placeThread.h"

#include <assert.h>
#include "oaDesignDB.h"

using std::vector;
using std::string;

int CapoPlugInCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    static_cast<void>(clientData);
    static_cast<void>(interp);
    static_cast<void>(argc);
    static_cast<void>(argv);

    if(argc < 4)
    {
        std::string message("Capo");
        message+=": run the Capo placer\n";
        message+="Usage: Capo [library] [cell] [view] &lt;additional Capo arguments&gt;\n"
            "\tlibrary - the OpenAccess library name\n"
            "\tcell    - the OpenAccess cell name\n"
            "\tview    - the OpenAccess view name\n";
        cout<<message;
        return TCL_OK;
    }

    CapoPlugIn* plugInPtr = static_cast<CapoPlugIn*>(clientData);

    try
    {
        std::string lib=argv[1];
        std::string cell=argv[2];
        std::string view=argv[3];

        cout<<"Starting The Capo Placer..."<<endl;
        CapoPIThread * placer = new CapoPIThread(lib, cell, view, argc, argv);
        plugInPtr->getAPI()->giveThread(placer);
    }
    catch (oa::oaException &e) 
    {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        return TCL_ERROR;
    }

    return TCL_OK;
}

int RunCapoPlugInCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    static_cast<void>(clientData);
    static_cast<void>(interp);
    static_cast<void>(argc);
    static_cast<void>(argv);

    CapoPlugIn* plugInPtr = static_cast<CapoPlugIn*>(clientData);

    try
    {
        string lib, cell, view;
        plugInPtr->getAPI()->getActiveDesign(lib, cell, view);

        string capo_command = "capo " + lib + " " + cell + " " + view ;

        plugInPtr->getAPI()->runCommand(capo_command);
    }
    catch (oa::oaException &e) 
    {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        return TCL_ERROR;
    }

    return TCL_OK;
}

int RunCapoOptionsPlugInCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    static_cast<void>(clientData);
    static_cast<void>(interp);
    static_cast<void>(argc);
    static_cast<void>(argv);

    CapoPlugIn* plugInPtr = static_cast<CapoPlugIn*>(clientData);


    try
    {
        RunCapoOptionsPIT * runCapoOptionsDialog = new RunCapoOptionsPIT(0, "runCapoOptions", TRUE);

        string lib, cell, view;
        plugInPtr->getAPI()->getActiveDesign(lib, cell, view);
        //cout<<"DPDEBUG: Capo plugin, got design: "<<lib<<"/"<<cell<<"/"<<view<<endl;
        runCapoOptionsDialog->setActiveDesign(lib.c_str(), cell.c_str(), view.c_str());

        plugInPtr->getAPI()->connectCommand(runCapoOptionsDialog, SIGNAL( sigRunCapo(const std::string& ) )
                );
        plugInPtr->getAPI()->giveDialog(runCapoOptionsDialog);

    }
    catch (oa::oaException &e) 
    {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
    }

    return TCL_OK;
}

oagBazaar::oagBazaarPlugInInterface * newPlugIn(void)
{
    CapoPlugIn* plugIn = new CapoPlugIn; //memory leak!
    return plugIn;
}

void CapoPlugIn::load(std::auto_ptr<oagBazaar::oagBazaarPlugInAPI> bazaarAPI)
{
   //DO THIS FIRST BEFORE ANYTHING ELSE 
   _bazaarAPI = bazaarAPI; //take ownership of the API object pointer

   _bazaarAPI->registerTCLCommand(&CapoPlugInCmd, "capo", this);
   _bazaarAPI->registerTCLCommand(&RunCapoPlugInCmd, "run_capo", this);
   _bazaarAPI->registerTCLCommand(&RunCapoOptionsPlugInCmd, "run_capo_options", this);

   QPopupMenu* capoPIMenu = _bazaarAPI->addMenu("Capo");
   QToolBar* capoToolBar = _bazaarAPI->addToolBar("Capo Tool Bar");

   QAction* runCapoAction = _bazaarAPI->createAction("run_capo", "Run Capo", "Run Capo.", "designer_editgrid.png");
   QAction* runCapoOptionsAction = _bazaarAPI->createAction("run_capo_options", "Run Capo with options", "Run Capo with options.", "designer_editgrid.png");

   _bazaarAPI->addMenuItem(capoPIMenu, runCapoAction);
   _bazaarAPI->addMenuItem(capoPIMenu, runCapoOptionsAction);
   _bazaarAPI->addToolBarButton(capoToolBar, runCapoAction);
   _bazaarAPI->addToolBarButton(capoToolBar, runCapoOptionsAction);

}



