# Variations of the greedy sizing method:
# o greedySizingRecoverPower - feasible start greedy sizing
# o greedySizingPower - essentially an adaptation of TILOS, 
#        an infeasible start greedy method. May not return a timing
#        feasible result
# o greedySizingTiming - Uses a TILOS like method to optimize timing.
#        optimizes the gate on critical path with best delay/ power tradeoff

proc greedySizingRecoverPower { } {
    #Performs a greedy sensitivity based sizing
    puts "Running greedySizingRecoverPower" 
    set fid [open "greedySizingRecoverPower.log" "w"]
    
    setDontUse * false
    set_dont_touch [get_cells] false
    
    setEcoMode -refinePlace false
    
    set cl "(direction == out)"
    set pcoll [get_pins -filter $cl]
    
    set sensList {}
    writeIterateInfo "Initial" $fid
    
    set loop 1
    while {$loop} {
      set loop 0
    # Write comparison scripts
    foreach_in_collection iPin $pcoll {
        
        if {![get_property $iPin is_clear] && ![get_property $iPin is_async] && ![get_property $iPin is_clock]} {
            set hname [get_property $iPin hierarchical_name]
            set ppname [get_property $iPin hierarchical_name]
            #get the gate associated with the pin
            regsub -all {/} $hname { } hname
            scan $hname "%s %s" gname pname
            set cgate [get_cells $gname]
            
            set cellname0 [get_property $cgate ref_lib_cell_name]
            set iscom [get_property $cgate is_combinational]
            
            # get the slack
            set s0 [min [get_property $iPin slack_max_rise]  [get_property $iPin slack_max_fall]]
            
            if {($s0 != "NA") && ($s0 != "INFINITY") && ($s0 >= 0) } {
                # get the initial leakage
                
                set leak0 [getLeakagePowerLibCell $cellname0 ]  
                set lbool 1
                if {$iscom} {
                    set opts [evaluateOptions $gname]
                    
                    foreach opt $opts { 
                        set cellnameN [lindex $opt 0]
                        set DS [lindex $opt 1]
                        set DP [lindex $opt 2]
                        
                        if { $DP < 0 } {
                            if [expr $DS <= 0] {
                                set sens [expr "-$DS*$DP"]
                            } else {
                                set sens [expr "-$DP/$DS"]
                            }
                            lappend sensList [list $gname $ppname $cellnameN $DP $DS $sens]
                        }
                    }
                }
                
            }
            
        }
    }
    
    puts "************* SIZING ***************"
    
    set sensList [lsort -real -index 5 -decreasing $sensList]
    
    set nReject 0
    
    set ind 0
    while { $ind < [llength $sensList] } {
        set st [lindex $sensList $ind]
        set gname [lindex $st 0]
        set ipin [get_pin [lindex $st 1]]
        set cellnameN [lindex $st 2]
        set sens [lindex $st 5]
        
        set cgate [get_cells $gname]
        set cellname0 [get_property $cgate ref_lib_cell_name]
        
        if { [getLeakagePowerLibCell $cellnameN ] < [getLeakagePowerLibCell $cellname0] } {
          set changed 1
          ecoChangeCell -inst $gname -cell $cellnameN
        
          if {[greedyLocalSlack $gname] < 0.0} {
              set changed 0
              ecoChangeCell -inst $gname -cell $cellname0
              #puts "    - move rejected (TNS)"
          }
          
          if {$changed} {
	    set loop 1
            puts "Changed $gname from $cellname0 -> $cellnameN ($sens)"
            set nReject 0
            # update the sensitivities
            set sensList [lrange $sensList [expr $ind + 1] end]
            set mcl [all_fanin -levels 1 -only_cells -to ${gname}/*]
            # find a 5 level region around it
            append_to_collection mcl [all_fanout -levels 1 -only_cells -from ${gname}/*] -unique
            
            # not too efficient, but not important
            foreach_in_collection cg $mcl {
              set ngname [get_property $cg hierarchical_name]
              #puts "   --- Adjusting $ngname"
              
              set ii 0
              while { $ii < [llength $sensList] } {
                set st [lindex $sensList $ii]
                set sgname [lindex $st 0]
                if {$sgname == $ngname} {
                  # update this sensitivity
                  set cellnameC [lindex $st 2]
                  set pnameC    [lindex $st 1]
                  set cp        [get_pin $pnameC]
                  
                  set cellname0 [get_property $cg ref_lib_cell_name]
          
		  set opt [evaluateOption $sgname $cellnameC]
                  
		  set DS [lindex $opt 0]
		  set DP [lindex $opt 1]
		  
                  if [expr $DS <= 0] {
                    set sens [expr "-$DS*$DP"]
                  } else {
                    set sens [expr "-$DP/$DS"]
                  }
                                    
                  set sensList [lreplace $sensList $ii $ii [list $sgname $pnameC $cellnameC $DP $DS $sens] ]
                  set sensList [lsort -real -index 5 -decreasing $sensList]
                  #puts "     -------- $sgname $cellname0 -> $cellnameC " 
                }
                incr ii 1
              }
            }
            set ind -1
          } else {
            incr nReject 1
          }
           
          
        }
        if {$nReject > 100} {
          break
        }
        
        incr ind 1
    }
    }
    close $fid
}

proc greedySizingPower { } {
    puts "Running greedySizingPower" 
    encMessage info 0
    #Performs a greedy sensitivity based sizing
    set fid [open "greedySizingPower.log" "w"]
    
    setDontUse * false
    set_dont_touch [get_cells] false
    
    # Don't do incremental placements inbetween cell sizings
    setEcoMode -refinePlace false
    
    set smin -1000
    set iters 0

    # iterate until the design is feasible, or the 
    while { ([getMinSlack] < 0) && ($iters < 1000) } {
        incr iters 1
        
        set cl "(direction == out)"
        set pcoll [get_pins -filter $cl]

        set smin 10000
        # find minimum slack
        foreach_in_collection iPin $pcoll {
            set s0 [getSlackPin $iPin]
            if {($s0 != "NA") && ($s0 != "INFINITY") && ($s0 < $smin)} {
                set smin $s0
            }
        }

        puts "$iters : $smin"

        set sensList {}
        
        # Make sensitivity list
        foreach_in_collection iPin $pcoll {
            if {![get_property $iPin is_clear] && ![get_property $iPin is_async] && ![get_property $iPin is_clock]} {
                set hname [get_property $iPin hierarchical_name]
                set ppname [get_property $iPin hierarchical_name]
                #get the gate associated with the pin
                regsub -all {/} $hname { } hname
                scan $hname "%s %s" gname pname
                set cgate [get_cells $gname]
                set cellname0 [get_property $cgate ref_lib_cell_name]
                set iscom [get_property $cgate is_combinational]
                
                # get the slack
                set s0 [getSlackPin $iPin]
                
                # only change the negative slack minimum slack instances
                if {($s0 != "NA") && ($s0 != "INFINITY") && ($s0 < 0) && ($s0 == $smin) } {
                    # get the initial leakage
                    set leak0 [getLeakagePowerLibCell $cellname0]
                    
                    set lbool 1
                    if {$iscom} {

                        set opts [evaluateOptions $gname]
                        
                        foreach opt $opts { 
                        set cellnameN [lindex $opt 0]
                            set DS [lindex $opt 1]
                            set DP [lindex $opt 2]
                            
                            if { $DS > 0 } {
                                if {$DP > 0} {
                                    set sens [expr "$DS/$DP"]
                                } else {
                                    set sens [expr "-$DS*$DP"]
                                }
                                lappend sensList [list $gname $ppname $cellnameN $DP $DS $sens]
                            }
                        }
                    }
                }
            }
        }
        
        puts "************* SIZING ***************"
        
        set sensList [lsort -real -index 5 -decreasing $sensList]
        set nChanges 0
        foreach st $sensList {
            set gname [lindex $st 0]
            set ipin [get_pin [lindex $st 1]]
            set cellnameN [lindex $st 2]

            set s0 [getSlackPin $ipin]
            

            
            #puts $gname
            set cgate [get_cells $gname]
            set cellname0 [get_property $cgate ref_lib_cell_name]
            
            ecoChangeCell -inst $gname -cell $cellnameN
            
            set sN [getSlackPin $ipin]
            if {$s0 >= $sN } {
                ecoChangeCell -inst $gname -cell $cellname0
                #puts "    - move rejected (SD)"
            } else {
                incr nChanges 1
		puts $fid "$iters : $smin $gname $cellname0 -> $cellnameN"
                break
            }
            
        }
	if {! $nChanges } {
	    puts "No changes made, ending greedySizingPower"
	    break
	}
    }
    #refinePlace
    #ecoPlace
    #ecoRoute

    close $fid
    #encMessage info 1
}

proc greedySizingTiming { } {
    puts "Running greedySizingTiming" 

    set sm0 [saveCurrentSizing]
    set ss0 [getMinSlack]


    #Performs a greedy sensitivity based sizing
    set fid [open "greedySizingTiming.log" "w"]
    
    setDontUse * false
    set_dont_touch [get_cells] false
    
    # Don't do incremental placements inbetween cell sizings
    setEcoMode -refinePlace false
    
    set smin -1000
    set iters 0
    set smin0 [getMinSlack]
    set incrcounter 0
    set loopf 1
    # iterate until the design is feasible, or the 
    while { ($loopf) && ($iters < 200) } {
        incr iters 1
        
        set cl "(direction == out)"
        set pcoll [get_pins -filter $cl]

        set smin [getMinSlack]

        puts $fid "$iters : $smin"
        puts " ******* $iters : $smin *************"
        set sensList {}
        
        # Make sensitivity list
        foreach_in_collection iPin $pcoll {
            if {![get_property $iPin is_clear] && ![get_property $iPin is_async] && ![get_property $iPin is_clock]} {
                set hname [get_property $iPin hierarchical_name]
                set ppname [get_property $iPin hierarchical_name]
                #get the gate associated with the pin
                regsub -all {/} $hname { } hname
                scan $hname "%s %s" gname pname
                set cgate [get_cells $gname]
                set cellname0 [get_property $cgate ref_lib_cell_name]
                set iscom [get_property $cgate is_combinational]
                
                # get the slack
                set s0 [getSlackPin $iPin]
                
                # only change the negative slack minimum slack instances
                if {($s0 != "NA") && ($s0 != "INFINITY")  && ($s0 == $smin) } {
                    # get the initial leakage


                    #puts "Examining $hname $cellname0"
                    set leak0 [getLeakagePowerLibCell $cellname0]
                    
                    set lbool 1
                    if {$iscom} {
                        set opts [evaluateOptions $gname]
                        
                        foreach opt $opts { 
                            set cellnameN [lindex $opt 0]
                            set DS [lindex $opt 1]
                            set DP [lindex $opt 2]
                            
                            if { $DS > 0 } {
                                if {$DP > 0} {
                                    set sens [expr "$DS/$DP"]
                                } else {
                                    set sens [expr "-$DS*$DP"]
                                }
                                lappend sensList [list $gname $ppname $cellnameN $DP $DS $sens]
                            }
                        }
                    }
                }
            }
        }
        set nopts [llength $sensList]
        if {$nopts == 0 } {
            puts "No change options!"
            puts $fid "break - no options"
            break
            
        } 
        puts "************* SIZING ${nopts} ***************"
        
        set sensList [lsort -real -index 5 -decreasing $sensList]
        set nChanges 0
        foreach st $sensList {
            set gname [lindex $st 0]
            set ipin [get_pin [lindex $st 1]]
            set cellnameN [lindex $st 2]

            set s0 [getSlackPin $ipin]
            
            #puts $gname
            set cgate [get_cells $gname]
            set cellname0 [get_property $cgate ref_lib_cell_name]
            
            ecoChangeCell -inst $gname -cell $cellnameN
            
            set sN [getSlackPin $ipin]
            if {$s0 >= $sN } {
                ecoChangeCell -inst $gname -cell $cellname0
            } else {
                incr nChanges 1
                puts $fid "$iters : $smin $gname $cellname0 -> $cellnameN"
                break
            }
            
        }
        if { $nChanges == 0 } {
            puts $fid "break - no cells changed"
            break
        }
        if {$smin >= [getMinSlack] } {
            incr incrcounter 1
        } else {
            set incrcounter 0
        }
        if {$incrcounter > 3 } {
            puts $fid "incr counter exceeds 3"
            break
        }
    }
    #refinePlace
    #ecoPlace
    #ecoRoute
    set smin [getMinSlack]

    if {$ss0 > $smin} {
        restoreSizing $sm0
        puts "$ss0 > $smin, Reverting sizes..."
    }

    set smin [getMinSlack]
    puts "Slack $smin0 -> $smin"
    close $fid
}

# Computes the slack a neighborhood around the current gate.
# This is to ensure that no timing violations are present
proc greedyLocalSlack { gn } {
  set cpin [getOutputPin $gn]
  
  set smin 1000
  set dcc  [all_fanin -pin_levels 3 -to $cpin ]
  foreach_in_collection dc $dcc {
    set sc [getSlackPin $dc]
    if { $sc < $smin } {
      set smin $sc
    }
  }
  set dcc  [all_fanout -pin_levels 3 -from $cpin ]
  foreach_in_collection dc $dcc {
    set sc [getSlackPin $dc]
    if { $sc < $smin } {
      set smin $sc
    }
  }
  return $smin
}
