#include "ManufacturingParams.h"
#include "fftInterface.h"
#include "FeatureExtraction.h"

using namespace std;
//using namespace boost::polygon::operators;
//using namespace boost::polygon;
typedef boost::polygon::polygon_90_with_holes_data<double> boostPolygon;
typedef boost::polygon::polygon_90_data<double> boostPolygon_withoutHoles;
typedef Eigen::MatrixXcf MatrixXcf;
typedef Eigen::MatrixXf MatrixXf;
typedef Eigen::VectorXf VectorXf;

FeatureExtraction::FeatureExtraction(const int &pSize, const int &sX, const int &sY, const int &cX, const int &cY){
pixelSize = pSize;
shiftX = sX;
shiftY = sY;
chipX = cX;
chipY = cY;
}


//*********************************************************
//dimReduction(): reduces the dimension of autocorrelation matrix using FFT and picking fewer orders
//*********************************************************
void FeatureExtraction::dimReductionFFT(const vector<Eigen::MatrixXcf> &allAutoCorrelation, const int &numFeatures, VectorXf &features) const{

features.resize(numFeatures);
int featureCount = 0;
features(featureCount) = 1.0;
featureCount++;
int numOrders = (int)sqrt((numFeatures-1)/allAutoCorrelation.size());

for (vector<MatrixXcf>::const_iterator a = allAutoCorrelation.begin(); a != allAutoCorrelation.end(); a++){
	int autoCrows = a->rows();
	int autoCcols = a->cols();
	MatrixXcf fullFT(autoCrows, autoCcols);

	fftInterface::getFourier(*a, fullFT);
	int totOrders = fullFT.rows()*fullFT.cols();
	for (int c = 0; c < numOrders; c++){
		for (int r = 0; r < numOrders; r++){
			features(featureCount) = abs(fullFT(r, c))/((float) totOrders);
//		cout<<features(featureCount)<<" ";
			featureCount++;
		}
	}
}

return;
}

//***************************************************************************************
//Dimension reduction by DCT transform
//**************************************************************************************
void FeatureExtraction::dimReductionDCT(const vector<MatrixXcf> &allAutoCorrelation, const int &numFeatures, VectorXf &features) const{

features.resize(numFeatures);
int featureCount = 0;
features(featureCount) = 1.0;
featureCount++;
int numOrders = (int)sqrt((numFeatures-1)/allAutoCorrelation.size());

for (vector<MatrixXcf>::const_iterator a = allAutoCorrelation.begin(); a != allAutoCorrelation.end(); a++){
	int autoCrows = a->rows();
	int autoCcols = a->cols();
	cout<<"Size of matrix: "<<autoCrows<<"x"<<autoCcols<<endl;
	MatrixXf fullDCT(autoCrows, autoCcols);
	fftInterface::getDCT(*a, fullDCT);
	int totOrders = fullDCT.rows()*fullDCT.cols();
	for (int c = 0; c < numOrders; c++){
		for (int r = 0; r < numOrders; r++){
			features(featureCount) = (fullDCT(r, c))/((float) totOrders);
//		cout<<features(featureCount)<<" ";
			featureCount++;
		}
	}
}

return;
}
//****************************************************************************
//readPCAfile(): Reads the mean and transform matrix for the PCA
//***************************************************************************
void FeatureExtraction::readPCAfile(const string &pcaFile, const int &numFeatures, MatrixXf &PC, VectorXf &M) const{

ifstream fin(pcaFile.c_str());
string line;
bool PCmode = false;
int row = 0;
if (fin.is_open()){
   while (fin.good()){
      getline(fin, line);
      if (line[0] != '#'){
         char delimiter=':';
         int pos = line.find(delimiter);
         string field = line.substr(0, pos);
         string value = line.substr(pos + 1);
         if (field == "Mean"){
                vector<string> meanS;
                ManufacturingParams::getWords(value, meanS, " ");
//		cout<<"Size of meanS: "<<meanS.size()<<endl;
                for (int x = 0; x < meanS.size(); x++) M(x) = atof(meanS[x].c_str());
          }
         else if (field == "PC"){
            PCmode = true;
            continue;
         }
        else if (PCmode){
           vector<string> PCvals;
           ManufacturingParams::getWords(line, PCvals, ",");
//	   cout<<"Number of cols in PC: "<<PCvals.size()<<endl;
	   if (PCvals.size() == numFeatures){ 
		for (int s = 0; s < PCvals.size(); s++){ 
			PC(row, s) = atof(PCvals[s].c_str());
		}
           	row++;
	   }
//	   else cout<<"Error: Number of row entries incorrect for this line: "<<line<<endl;	
        }
      }
   }
}

cout<<"Number of rows of PC: "<<row<<endl;


return;

}
//******************************************************************************
//dimReductionPCA():
//*****************************************************************************
void FeatureExtraction::dimReductionPCA(const MatrixXcf &autoCorrelation, const string &pcaFile, const int &numFeatures, VectorXf &features) const{

VectorXf fullFeature(autoCorrelation.rows()*autoCorrelation.cols());
int count = 0;
for (int r = 0; r < autoCorrelation.rows(); r++){
	for (int c = 0; c < autoCorrelation.cols(); c++){
		fullFeature(count++) = real(autoCorrelation(r, c));
	}
}

//cout<<"Full feature vector: "<<fullFeature<<endl;

MatrixXf A(count, numFeatures);
VectorXf M(count);
cout<<"Number of features: "<<numFeatures<<endl;
readPCAfile(pcaFile, numFeatures, A, M);
//cout<<"Mean read in:"<<M<<endl;
cout<<"A: "<<endl;
for (int r = 0; r < A.rows(); r++){
	for (int c = 0; c < A.cols(); c++){
//		cout<<A(r, c)<<", ";
		if (isnan(A(r, c))) cout<<"Error NAN at Position: ("<<r<<","<<c<<"):  "<<A(r, c)<<endl;
	}
//	cout<<endl;
}

features.resize(numFeatures);
features = (fullFeature - M)*A;

return;
}


//**************************************************************************
//dimReductionNone():Just vectorizes the autocorrelation matrix and returns it as a feature vector (Allows applying PCA-like methods later)
//**************************************************************************
void FeatureExtraction::dimReductionNone(const MatrixXcf &autoCorrelation, VectorXf &features) const{

int rows = autoCorrelation.rows();
int cols = autoCorrelation.cols();
features.resize(rows*cols);
int count = 0;
for (int r = 0; r < rows; r++){
	for (int c = 0; c < cols; c++){
		features(count) = real(autoCorrelation(r, c));
		count++;
	}
}

return;	
}

//*******************************************************************************
//dimReductionIE(): Very simple with just 2 features, one is density (zeroth entry of autocorrelation matrix) and second 
//is weighted sum of all other entries, where weight is based on number of occurrences of each shifted term in IE formulation
//*******************************************************************************
void FeatureExtraction::dimReductionIE(const MatrixXcf &autoCorrelation, VectorXf &features) const{

features.resize(3);

features(0) = real(autoCorrelation(0, 0));

int numRows = autoCorrelation.rows();
int numCols = autoCorrelation.cols();
float totFactor = (float)(numRows*numCols);
float sum = 0.0;
for (int r = 0; r < numRows; r++){
	for (int c = 0; c < numCols; c++){
		if (!(r==0 && c==0)){
			float f = ((float)(numRows - r)*(numCols - c));
			sum += (f/totFactor)*real(autoCorrelation(r, c));
		}
	}
}

features(1) = sum;
features(2) = 1.0;

return;
}

//******************************************************************************
//getAutoCorrelationEntry(): Returns autocorrelation matrix entry for given shift value (accounts for different defect sizes)
//******************************************************************************
float FeatureExtraction::getAutoCorrelationEntry(const vector<vector<defectType> > &allProhibitedRegionswOrient, const int &angleID) const{

float value = 0.0;
int defectTypeID = 0;
using namespace boost::polygon::operators;

for (vector<defectType>::const_iterator s = allProhibitedRegionswOrient[angleID].begin(); s != allProhibitedRegionswOrient[angleID].end(); s++){
	list<boostPolygon> shiftedProhibitedRegions = s->prohibitedRegions;
	move(shiftedProhibitedRegions, shiftX, shiftY);
	list<boostPolygon> solutionSpace;
	solutionSpace |= allProhibitedRegionswOrient[0][defectTypeID].prohibitedRegions;
	solutionSpace &= shiftedProhibitedRegions;
	value += ((float)area(solutionSpace))*(s->getProbability());
	solutionSpace.clear();
	defectTypeID++;
}

return value;
}

//*******************************************************************************
//getAutoCorrelationEntry_approx(): Approximates the process by doing it for one defect size only and estimating others
//******************************************************************************
float FeatureExtraction::getAutoCorrelationEntry_approx(const vector<defectType> &allProhibitedRegions) const{

using namespace boost::polygon::operators;
int last = allProhibitedRegions.size() - 1;
float largestPBprob = allProhibitedRegions[last].getProbability();
list<boostPolygon> largestPBregion = allProhibitedRegions[last].prohibitedRegions;
list<boostPolygon> largestShiftedPBregion = largestPBregion;
move(largestShiftedPBregion, shiftX, shiftY);
list<boostPolygon> solutionSpaceLargest;
solutionSpaceLargest |= largestPBregion;
solutionSpaceLargest &= largestShiftedPBregion;
//float areaLargest = (float) area(solutionSpaceLargest);

/*float peri = 0.0;
int totalCorners = 0;
int acuteEdges = 0;
float areaLargest = 0.0;
for (list<boostPolygon>::iterator b = solutionSpaceLargest.begin(); b != solutionSpaceLargest.end(); b++){
	int numPoints = b->size();
	int pointCount = 0;
	vector<boostPoint> points;
	points.insert(points.begin(), b->begin(), b->end());
	for (vector<boostPoint>::iterator p = points.begin(); p != points.end(); p++){
		boostPoint p1 = *p;
		boostPoint p2;
		if (pointCount != numPoints - 1) p2 = *(p+1);
		else p2 = *(points.begin());
		boostPoint pBefore;
		if (pointCount == 0) pBefore = *(points.end() - 1);
		else pBefore = *(p-1);
		boostPoint pAfter;
		if (pointCount == numPoints - 1) pAfter = *(points.begin() + 1);
		else if (pointCount == numPoints - 2) pAfter = *(points.begin());
		else pAfter = *(p+2);
		if (p1.x() == p2.x()){
			if ((pBefore.x() >= p1.x() && pAfter.x() >= p1.x()) || (pBefore.x() <= p1.x() && pAfter.x() <= p1.x())) acuteEdges++;
		}
		else if (p1.y() == p2.y()){
			if ((pBefore.y() >= p1.y() && pAfter.y() >= p1.y()) || (pBefore.y() <= p1.y() && pAfter.y() <= p1.y())) acuteEdges++;
		}
		else cout<<"Error: edge not rectilinear"<<endl;
		peri += (float)manhattan_distance(p1, p2);
		pointCount++;
	}
//	peri += perimeter(*b);	
	totalCorners += numPoints;
	areaLargest += ((float)area(*b)); 
}


cout<<"Area largest: "<<areaLargest<<" Perimeter: "<<peri<<" corners: "<<totalCorners<<endl;
*/


// hardcoded for now
float A = 3.0*0.5478*0.191/0.0471;
float B = -3.0*0.5478*0.094/0.0471;
float CDtol = 0.8; 
float fwhm = allProhibitedRegions[last].getHalfWidth();
float height = allProhibitedRegions[last].getHeight();
float outerRadiusLargest = round((0.5*fwhm)*sqrt(log(A*height) - log(CDtol - B)));
float innerRadiusLargest = round((0.5*fwhm)*sqrt(log(A*height) - log((2*CDtol) - B)));
float value = 0.0;
//cout<<"outerRadiusLargest: "<<outerRadiusLargest<<endl;
for (list<boostPolygon>::iterator b = solutionSpaceLargest.begin(); b != solutionSpaceLargest.end(); b++){
	value += (area(*b)*allProhibitedRegions[last].getProbability());
	for (vector<defectType>::const_iterator d = allProhibitedRegions.begin(); d != allProhibitedRegions.end()-1; d++){
	//get value of delta reduction assuming only outer radius matters for now
		fwhm = d->getHalfWidth();
		height = d->getHeight();
		float outerRadius = 0.0;
		if (A*height >= (CDtol - B)) outerRadius = round((0.5*fwhm)*sqrt(log(A*height) - log(CDtol - B)));
		float innerRadius = 0.0;
		if (A*height >= (2*CDtol - B)) innerRadius = round((0.5*fwhm)*sqrt(log(A*height) - log((2*CDtol) - B)));
		float scaleFactor = (outerRadius + innerRadius)/(outerRadiusLargest + innerRadiusLargest);
//		float delta = 0.5*(outerRadiusLargest - outerRadius) + 0.5*(innerRadiusLargest - innerRadius);
//		cout<<"fwhm: "<<fwhm<<"height: "<<height<<"outerRadius: "<<outerRadius<<" delta"<<delta<<endl;
//		value += (areaLargest - peri*delta + ((float)acuteEdges)*delta*delta)*(d->getProbability());
		boostPolygon bCopy = *b;
		value += (area(scale(bCopy, scaleFactor))*d->getProbability());
		
	}
}
//float smallestPBprob = allProhibitedRegions[0].getProbability();
//list<boostPolygon> smallestPBregion = allProhibitedRegions[0].prohibitedRegions;
//list<boostPolygon> smallestShiftedPBregion = smallestPBregion;
//move(smallestShiftedPBregion, shiftX, shiftY);
//list<boostPolygon> solutionSpaceSmallest;
//solutionSpaceSmallest |= smallestPBregion;
//solutionSpaceSmallest &= smallestShiftedPBregion;
//float areaSmallest = (float) area(solutionSpaceSmallest);

//float lowerBound = ((1.0 - largestPBprob)*areaSmallest) + (largestPBprob*areaLargest);
//float upperBound = (smallestPBprob*areaSmallest) + ((1.0 - smallestPBprob)*areaLargest);

return value;
//float value = 0.0;
//for (list<boostPolygon>::iterator b = solutionSpace.begin(); b != solutionSpace.end(); b++){
//	int per = perimeter(*b);
//}

}

//*******************************************************************************
//rotatePolygon(): Rotates given input polygon
//******************************************************************************
void FeatureExtraction::rotatePolygon(const float &angle, const vector<boostPoint> &points, vector<boostPoint> &newPoints) const{
int xOff = chipX/2;
int yOff = chipY/2;
float angleRad = 3.14*angle/180.0;
for (vector<boostPoint>::const_iterator p = points.begin(); p != points.end(); p++){
	float xOldShifted = (float)(p->x()-xOff);
	float yOldShifted = (float)(p->y()-yOff);
	float xNew = xOldShifted*cos(angleRad) - yOldShifted*sin(angleRad);
	float yNew = xOldShifted*sin(angleRad) + yOldShifted*cos(angleRad);
	int xNewUp = (int)(xNew) + xOff;
	int yNewUp = (int)(yNew) + yOff;
	boostPoint pNew;
	pNew.x(xNewUp);
	pNew.y(yNewUp);
	newPoints.push_back(pNew);
}
return;
}
//********************************************************************************
//getRotatedLayout(): Rotates the layout by the given angle (in degrees)
//********************************************************************************
void FeatureExtraction::getRotatedLayout(const vector<defectType> &allProhibitedRegionsBefore, const float &angle, vector<defectType> &allProhibitedRegionsAfter) const{
int defectTypeID = 0;
for (vector<defectType>::const_iterator s = allProhibitedRegionsBefore.begin(); s != allProhibitedRegionsBefore.end(); s++){
	allProhibitedRegionsAfter.push_back(*s);
	for (list<boostPolygon>::iterator b = allProhibitedRegionsAfter[defectTypeID].prohibitedRegions.begin(); b != allProhibitedRegionsAfter[defectTypeID].prohibitedRegions.end(); b++){
//		handle holes and coordinates of polygon
		vector<boostPoint> points;
		vector<boostPoint> newPoints;
		points.insert(points.begin(), b->begin(), b->end());
		rotatePolygon(angle, points, newPoints);
		b->set(newPoints.begin(), newPoints.end());
//		allHoles.insert(allHoles.begin(), begin_holes(*b), end_holes(*b));
//		list<boostPoint>::iterator p = begin_holes(*b);
		list<boostPolygon_withoutHoles> allHolesNew;
		for (list<boostPolygon_withoutHoles>::const_iterator h = begin_holes(*b); h != end_holes(*b); h++){
			vector<boostPoint> pointsH;
			vector<boostPoint> pointsHnew;
			pointsH.insert(pointsH.begin(), h->begin(), h->end());
			rotatePolygon(angle, pointsH, pointsHnew);
			boostPolygon_withoutHoles currentHole;
			currentHole.set(pointsHnew.begin(), pointsHnew.end());
			allHolesNew.push_back(currentHole);
		}
		b->set_holes(allHolesNew.begin(), allHolesNew.end());
	}
	defectTypeID++;
}

}
//********************************************************************************
//getFeatures_ideal(): computes autocorrelation matrix, and calls dimension reduction function to get features. Multiple size case handled exactly, Boolean operation repeated for each size
//********************************************************************************
void FeatureExtraction::GetFeatures(const vector<defectType> &allProhibitedRegions, const string &modelType, string pcaFile, const int &numFeatures, VectorXf &features, float &PBdensity) const{

//cout<<"Copied prohibited list size "<<shiftedProhibitedRegions.size()<<endl;
float reticleArea = ((float) chipX + shiftX)*((float) chipY + shiftY);
//Different angles
int numAngles = 4;
vector<float> allAngles(numAngles);
allAngles[0] = 0.0; 
allAngles[1] = 90.0; allAngles[2] = 180.0; allAngles[3] = 270.0;

vector<vector<defectType> > allProhibitedRegionswOrient(numAngles);
allProhibitedRegionswOrient[0] = allProhibitedRegions;
#pragma omp parallel for
for (int angleID = 0; angleID < numAngles; angleID++){
	vector<defectType> allProhibitedRegionsRot;
	getRotatedLayout(allProhibitedRegions, allAngles[angleID], allProhibitedRegionsRot);
	allProhibitedRegionswOrient[angleID] = allProhibitedRegionsRot;
}

if (modelType == "sparse"){
	int numRows = (int)floor(log((float)shiftY/(float)pixelSize)/log(2))+2;
	int numCols = (int)floor(log((float)shiftX/(float)pixelSize)/log(2))+2;
	int featureSize = numRows*numCols*numAngles;
	features.resize(featureSize);
	for (int angleID = 0; angleID < numAngles; angleID++){
		#pragma omp parallel for
		for (int r = 0; r < numRows; r++){
			for (int c = 0; c < numCols; c++){
				int currentShiftX = 0;
				if (c > 0) currentShiftX = (1 << (c-1))*pixelSize;
				int currentShiftY = 0;
				if (r > 0) currentShiftY = (1 << (c-1))*pixelSize;
				int index = r + c*numRows + angleID*numRows*numCols;
				features(index) = (getAutoCorrelationEntry(allProhibitedRegionswOrient, angleID))/(reticleArea);
			}
		}
	}
	PBdensity = features(0);	
}
else {

	int autoCrows = shiftX/pixelSize;
	int autoCcols = shiftY/pixelSize;
	complex<float> z(0.0, 0.0);
	MatrixXcf autoCorrelation(autoCrows, autoCcols);

	//StopWatch_parallel(1);
	//Compute autocorrelation using boolean operations
	vector<MatrixXcf> allAutoCorrelation;

	for (int angleID = 0; angleID < numAngles; angleID++){
	#pragma omp parallel for
	for (int r = 0; r < autoCrows; r++){
		for (int c = 0; c < autoCcols; c++){

//			autoCorrelation(r, c) = z;
			int currentShiftX = r*pixelSize;
			int currentShiftY = c*pixelSize;
			float normalizedValue = (getAutoCorrelationEntry(allProhibitedRegionswOrient, angleID))/(reticleArea);
			complex<float> T(normalizedValue, 0.0);
			autoCorrelation(r, c) = T;
//			cout<<"Row "<<r<<" Col "<<c<<" "<<T<<endl;
//			move(shiftedProhibitedRegions, -currentShiftX, -currentShiftY);
		}
	}
	allAutoCorrelation.push_back(autoCorrelation);
	}

	//cout<<"Runtime for autocorrelation matrix computation: "<<StopWatch_parallel(0)<<endl;
	//cout<<"Expected Prohibited region density (aggregrate over defect sizes):"<<autoCorrelation(0, 0)<<endl;
	PBdensity = real(allAutoCorrelation[0](0, 0));
	//get feature set by dimension reduction
	//StopWatch(1);
	cout<<"Size of one autocorrelation matrix: "<<allAutoCorrelation[0].rows()<<"x"<<allAutoCorrelation[0].cols()<<endl;
	if (modelType == "fft") dimReductionFFT(allAutoCorrelation, numFeatures, features);
	else if (modelType == "all") dimReductionNone(autoCorrelation, features);
	else if (modelType == "ie") dimReductionIE(autoCorrelation, features);
	else if (modelType == "pca") dimReductionPCA(autoCorrelation, pcaFile, numFeatures, features);
	else if (modelType == "dct") dimReductionDCT(allAutoCorrelation, numFeatures, features);
	else cout<<"Error: Model type undefined"<<endl;
	//cout<<"Runtime for dimension reduction: "<<StopWatch(0)<<endl;
	return;
}

}

